<?php
header("Content-Type: application/json");
require_once "../config/db.php";
require_once "../helpers/auth_middleware.php";
require_once "../helpers/guest_session.php";

$user_id = require_auth(false); 
$guest_id = null;

if (!$user_id) {
    $guest_id = get_guest_id(); // cria/retorna um ID de sessão
}


$raw = file_get_contents("php://input");
$data = json_decode($raw, true);

if (!is_array($data) || !isset($data["type"]) || !isset($data["question_id"])) {
    http_response_code(400);
    echo json_encode(["success" => false, "message" => "Payload inválido. Esperado { type, question_id }"]);
    exit;
}

$type = $data["type"]; // "fifty_fifty" ou "show_answer"
$question_id = intval($data["question_id"]);

if (!in_array($type, ["fifty_fifty", "show_answer"])) {
    http_response_code(400);
    echo json_encode(["success" => false, "message" => "Tipo de ajuda inválido"]);
    exit;
}

try {
    $pdo->beginTransaction();

    // Verifica se user existe
    $checkUser = $pdo->prepare("SELECT id FROM users WHERE id = ? FOR UPDATE");
    $checkUser->execute([$user_id]);
    if (!$checkUser->fetch()) {
        $pdo->rollBack();
        http_response_code(404);
        echo json_encode(["success" => false, "message" => "Usuário não encontrado"]);
        exit;
    }

    // Busca registro de helpers do usuário
    $stmt = $pdo->prepare("SELECT fifty_used, reveal_used FROM user_helpers WHERE user_id = ? FOR UPDATE");
    $stmt->execute([$user_id]);
    $helperRow = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$helperRow) {
        $insert = $pdo->prepare("INSERT INTO user_helpers (user_id, fifty_used, reveal_used) VALUES (?, 0, 0)");
        $insert->execute([$user_id]);
        $helperRow = ['fifty_used' => 0, 'reveal_used' => 0];
    }

    // Verifica se já usou a ajuda
    if ($type === "fifty_fifty" && intval($helperRow["fifty_used"]) === 1) {
        $pdo->rollBack();
        echo json_encode(["success" => false, "message" => "Ajuda 50/50 já utilizada"]);
        exit;
    }
    if ($type === "show_answer" && intval($helperRow["reveal_used"]) === 1) {
        $pdo->rollBack();
        echo json_encode(["success" => false, "message" => "Ajuda Mostrar Resposta já utilizada"]);
        exit;
    }

    // Buscar questão
    $q = $pdo->prepare("SELECT option_a, option_b, option_c, option_d, correct_option FROM questions WHERE id = ? LIMIT 1");
    $q->execute([$question_id]);
    $question = $q->fetch(PDO::FETCH_ASSOC);

    if (!$question) {
        $pdo->rollBack();
        http_response_code(404);
        echo json_encode(["success" => false, "message" => "Questão não encontrada"]);
        exit;
    }

    $response = [];

    if ($type === "fifty_fifty") {
        $options = ['A', 'B', 'C', 'D'];
        $correct = strtoupper($question['correct_option']);
        $wrongOptions = array_values(array_diff($options, [$correct]));
        shuffle($wrongOptions);

        $remove = array_slice($wrongOptions, 0, 2);
        $remaining = array_values(array_diff($options, $remove));

        $response = ['remove' => $remove, 'remaining' => $remaining];

        $upd = $pdo->prepare("UPDATE user_helpers SET fifty_used = 1 WHERE user_id = ?");
        $upd->execute([$user_id]);
    } elseif ($type === "show_answer") {
        $response = ["correct" => strtoupper($question["correct_option"])];

        $upd = $pdo->prepare("UPDATE user_helpers SET reveal_used = 1 WHERE user_id = ?");
        $upd->execute([$user_id]);
    }

    $pdo->commit();

    echo json_encode([
        "success" => true,
        "helper" => $type,
        "result" => $response
    ]);
} catch (PDOException $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    http_response_code(500);
    echo json_encode(["success" => false, "message" => "Erro no servidor: " . $e->getMessage()]);
    exit;
}
