<?php
header("Content-Type: application/json");
require_once "../config/db.php";
require_once "../helpers/guest_session.php";

// Tentar obter usuário logado
$user_id = null;
$headers = getallheaders();
if (isset($headers['Authorization'])) {
    require_once "../helpers/jwt_helper.php";
    $user_id = require_auth(true); // opcional
}

// Se não tiver usuário logado, criar guest_id
$guest_id = null;
if (!$user_id) {
    $guest_id = get_guest_id();
}

// Ler payload
$raw = file_get_contents('php://input');
$data = json_decode($raw, true);

if (!is_array($data) || !isset($data['answers']) || !is_array($data['answers'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Payload inválido.']);
    exit;
}

$group   = isset($data['group']) ? intval($data['group']) : null;
$answers = $data['answers'];

if (count($answers) === 0) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Nenhuma resposta enviada']);
    exit;
}

// 🔹 Validação: garantir que todas as questões pertencem ao grupo informado
$sentQuestionIds = [];
foreach ($answers as $a) {
    if (isset($a['question_id'])) $sentQuestionIds[] = intval($a['question_id']);
}

if (count($sentQuestionIds) > 0) {
    $in = implode(',', array_fill(0, count($sentQuestionIds), '?'));
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM questions WHERE id IN ($in) AND group_id = ?");
    $params = array_merge($sentQuestionIds, [$group]);
    $stmt->execute($params);
    $cnt = intval($stmt->fetchColumn());
    if ($cnt !== count($sentQuestionIds)) {
        http_response_code(400);
        echo json_encode(['success'=>false,'message'=>'Questões inválidas para essa fase']);
        exit;
    }
}

// Buscar respostas corretas
$qSelect = $pdo->prepare("SELECT id, correct_option FROM questions WHERE group_id = ?");
$qSelect->execute([$group]);
$questions = $qSelect->fetchAll(PDO::FETCH_ASSOC);

if (!$questions) {
    http_response_code(404);
    echo json_encode(['success' => false, 'message' => 'Nenhuma questão encontrada']);
    exit;
}

$correct_count = 0;
$wrong_count   = 0;
$results       = [];

foreach ($questions as $q) {
    $qid     = $q['id'];
    $correct = strtoupper($q['correct_option']);
    $given   = null;

    foreach ($answers as $ans) {
        if ($ans['question_id'] == $qid) {
            $given = strtoupper(substr(trim($ans['answer']), 0, 1));
            break;
        }
    }

    $is_correct = ($given === $correct);
    if ($is_correct) $correct_count++; else $wrong_count++;

    $results[] = [
        'question_id' => $qid,
        'given'       => $given,
        'correct'     => $correct,
        'is_correct'  => $is_correct
    ];
}

try {
    if ($user_id) {
        // 🔹 Fluxo se for usuário logado
        $pdo->beginTransaction();

        // Exemplo simples: 10 pts por acerto
        $score_delta = $correct_count * 10;

        // Atualizar score e grupo do usuário
        $upd = $pdo->prepare("UPDATE users 
                              SET score = score + ?, 
                                  current_group = current_group + 1, 
                                  answered = answered + ?,
                                  daily_answered = daily_answered + ?,
                                  last_answer_date = ?
                              WHERE id = ?");
        $upd->execute([$score_delta, $correct_count, $correct_count, gmdate("Y-m-d"), $user_id]);

        // (Opcional: inserir em leaderboard / attempts)

        $pdo->commit();

        echo json_encode([
            'success' => true,
            'mode'    => 'user',
            'user_id' => $user_id,
            'group'   => $group,
            'correct' => $correct_count,
            'wrong'   => $wrong_count,
            'score_delta' => $score_delta,
            'results' => $results
        ]);
    } else {
        // 🔹 Fluxo se for convidado → NÃO grava nada
        echo json_encode([
            'success'  => true,
            'mode'     => 'guest',
            'guest_id' => $guest_id,
            'group'    => $group,
            'correct'  => $correct_count,
            'wrong'    => $wrong_count,
            'results'  => $results,
            'message'  => $wrong_count === 0
                ? 'Parabéns! Avançou de fase.'
                : "Fim do jogo nesta fase. Você fez $correct_count acertos de " . count($questions)
        ]);
    }
} catch (PDOException $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Erro no servidor: ' . $e->getMessage()]);
    exit;
}
